<?php
// For development only — disable in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../db/db.php';
require_once __DIR__ . '/../jwt_utils/jwt_utils.php';

header('Content-Type: application/json');
$headers = getallheaders();
$method = $_SERVER['REQUEST_METHOD'];

// JWT Authentication
if (empty($headers['Authorization'])) {
    http_response_code(401);
    echo json_encode(['error' => 'No token!']); exit;
}
$jwt = str_replace('Bearer ', '', $headers['Authorization']);
$payload = validate_jwt($jwt, $jwt_secret);
if (!$payload || !isset($payload['id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Invalid token!']); exit;
}
$user_id = intval($payload['id']);

// NOTE: $conn should be available from your db.php (mysqli instance).
// If your db.php returns a Database object, make sure $conn is set accordingly.
if (!isset($conn) || !$conn) {
    // try to create via Database class if defined
    if (class_exists('Database')) {
        $dbObj = new Database();
        $conn = $dbObj->getConnection();
    }
    if (!isset($conn) || !$conn) {
        http_response_code(500);
        echo json_encode(['error' => 'Database connection not found']);
        exit;
    }
}

// --- Add Payment ---
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'payment_add') {

    // Permission: frontend already checks can_add
    // Backend safety check (basic)
    if (!isset($payload['id'])) {
        http_response_code(403);
        echo json_encode(['error' => 'Unauthorized']);
        exit;
    }

    $input = json_decode(file_get_contents("php://input"), true);

    $booking_id = intval($input['booking_id'] ?? 0);
    $amount     = floatval($input['amount'] ?? 0);
    $mode       = $conn->real_escape_string($input['mode'] ?? '');
    $remarks    = $conn->real_escape_string($input['remarks'] ?? '');

    if ($booking_id <= 0 || $amount <= 0 || empty($mode)) {
        echo json_encode(['error' => 'Invalid payment data']);
        exit;
    }

    // Check booking exists
    $chk = $conn->query("SELECT id FROM bookings WHERE id=$booking_id");
    if (!$chk || $chk->num_rows === 0) {
        echo json_encode(['error' => 'Booking not found']);
        exit;
    }

    $stmt = $conn->prepare("
        INSERT INTO booking_payments
        (booking_id, amount, payment_mode, remarks, paid_by)
        VALUES (?, ?, ?, ?, ?)
    ");
    $stmt->bind_param(
        "idssi",
        $booking_id,
        $amount,
        $mode,
        $remarks,
        $user_id
    );

    if ($stmt->execute()) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['error' => $stmt->error]);
    }

    $stmt->close();
    exit;
}



// --- Create Booking ---
if ($method === 'POST' && !isset($_GET['booking_id'])) {
    $input = json_decode(file_get_contents("php://input"), true);

    $purpose = $conn->real_escape_string($input['purpose'] ?? '');
    $start_time = $conn->real_escape_string($input['start_time'] ?? '');
    $end_time = $conn->real_escape_string($input['end_time'] ?? '');
    $type = $conn->real_escape_string($input['type'] ?? 'external'); // internal or external

    // --- basic validation
    if (empty($start_time) || empty($end_time)) {
        echo json_encode(['error' => 'Missing start_time or end_time']);
        exit;
    }

    // --- Check for slot conflicts ---
    $conflict_sql = "SELECT id FROM bookings
                    WHERE ((start_time < ?) AND (end_time > ?))
                    AND status = 'booked' LIMIT 1";
    $stmt_conf = $conn->prepare($conflict_sql);
    $stmt_conf->bind_param('ss', $end_time, $start_time);
    $stmt_conf->execute();
    $res_conf = $stmt_conf->get_result();
    if ($res_conf && $res_conf->fetch_assoc()) {
        echo json_encode(['error' => 'Slot not available']); exit;
    }
    $stmt_conf->close();

    // Begin transaction
    $conn->begin_transaction();

    try {
        // Insert booking
        $ins_sql = "INSERT INTO bookings (booked_by, purpose, start_time, end_time, status, type, created_at)
                    VALUES (?, ?, ?, ?, 'booked', ?, NOW())";
        $stmt = $conn->prepare($ins_sql);
        $stmt->bind_param('issss', $user_id, $purpose, $start_time, $end_time, $type);
        if (!$stmt->execute()) {
            throw new Exception("Booking insert failed: " . $stmt->error);
        }
        $booking_id = $conn->insert_id;
        $stmt->close();

        // -------------------------------------------------------
        // Determine slot duration in hours and fetch slot price
        // -------------------------------------------------------
        $start_ts = strtotime($start_time);
        $end_ts = strtotime($end_time);
        $duration_hours = 0;
        if ($start_ts !== false && $end_ts !== false && $end_ts > $start_ts) {
            $duration_hours = ($end_ts - $start_ts) / 3600;
        } else {
            // fallback: try to compute based on provided 'slot' in input
            $duration_hours = intval($input['slot'] ?? 0);
        }

        // attempt to find a slots record by slot_duration = duration_hours
        $slot_price = 0.0;
        if ($duration_hours > 0) {
            $slot_stmt = $conn->prepare("SELECT price FROM slots WHERE slot_duration = ? LIMIT 1");
            $dur_str = (string)$duration_hours; // bind as string to match stored values like "3"
            $slot_stmt->bind_param('s', $dur_str);
            $slot_stmt->execute();
            $slot_res = $slot_stmt->get_result();
            if ($slot_row = $slot_res->fetch_assoc()) {
                $slot_price = floatval($slot_row['price']);
            }
            $slot_stmt->close();
        }

        // -------------------------------------------------------
        // Handle services (input may be array of IDs OR full objects)
        // -------------------------------------------------------
        $service_items = []; // array of [service_name, price, quantity]
        $total_service_amount = 0.0;

        if (!empty($input['services']) && is_array($input['services'])) {
            // detect whether elements are numeric IDs or objects
            $are_ids = true;
            foreach ($input['services'] as $it) {
                if (!is_numeric($it)) { $are_ids = false; break; }
            }

            if ($are_ids) {
                // fetch service rows by ids
                $ids = array_map('intval', $input['services']);
                if (count($ids) > 0) {
                    $ids_list = implode(',', $ids);
                    // safe since we cast to int above
                    $srv_sql = "SELECT id, service_name, price FROM services WHERE id IN ($ids_list)";
                    $srv_result = $conn->query($srv_sql);
                    if ($srv_result) {
                        while ($r = $srv_result->fetch_assoc()) {
                            $name = $r['service_name'];
                            $price = floatval($r['price']);
                            $qty = 1;
                            $service_items[] = ['service_name' => $name, 'price' => $price, 'quantity' => $qty];
                            $total_service_amount += $price * $qty;
                        }
                    }
                }
            } else {
                // assume array of objects { service_name, quantity, price } or similar
                foreach ($input['services'] as $svc) {
                    $name = $conn->real_escape_string($svc['service_name'] ?? ($svc['name'] ?? ''));
                    $qty = intval($svc['quantity'] ?? 1);
                    $price = floatval($svc['price'] ?? 0);
                    if ($name !== '') {
                        $service_items[] = ['service_name' => $name, 'price' => $price, 'quantity' => $qty];
                        $total_service_amount += $price * $qty;
                    }
                }
            }
        }

        // -------------------------------------------------------
        // Compute total amount (slot + services)
        // -------------------------------------------------------
        $total_amount = floatval($slot_price) + floatval($total_service_amount);

        // -------------------------------------------------------
        // Generate next receipt number (6 digits padded)
        // -------------------------------------------------------
        $last_receipt = null;
        $r = $conn->query("SELECT receipt_number FROM booking_receipts ORDER BY id DESC LIMIT 1");
        if ($r && $r->num_rows > 0) {
            $row = $r->fetch_row();
            $last_receipt = $row[0] ?? null;
        }
        if ($last_receipt !== null && $last_receipt !== '') {
            $next_num = str_pad((string)(intval($last_receipt) + 1), 6, "0", STR_PAD_LEFT);
        } else {
            $next_num = "000001";
        }
        $receipt_number = $next_num;
        $receipt_pdf_path = "pdf-" . $booking_id . "-" . $receipt_number . ".pdf";

        // -------------------------------------------------------
        // Insert booking_receipts
        // -------------------------------------------------------
        $rec_stmt = $conn->prepare("INSERT INTO booking_receipts (booking_id, receipt_number, total_amount, receipt_pdf_path, created_at) VALUES (?, ?, ?, ?, NOW())");
        // types: booking_id (i), receipt_number (s), total_amount (d), receipt_pdf_path (s)
        $rec_stmt->bind_param('isds', $booking_id, $receipt_number, $total_amount, $receipt_pdf_path);
        if (!$rec_stmt->execute()) {
            throw new Exception("Insert receipt failed: " . $rec_stmt->error);
        }
        $receipt_id = $conn->insert_id;
        $rec_stmt->close();

        // -------------------------------------------------------
        // Insert booking_services
        // -------------------------------------------------------
        if (count($service_items) > 0) {
            $ins_srv_stmt = $conn->prepare("INSERT INTO booking_services (booking_id, service_name, quantity, price) VALUES (?, ?, ?, ?)");
            foreach ($service_items as $si) {
                $svc_name = $si['service_name'];
                $svc_q = intval($si['quantity']);
                $svc_p = floatval($si['price']);
                // types: booking_id (i), service_name (s), quantity (i), price (d)
                $ins_srv_stmt->bind_param('isid', $booking_id, $svc_name, $svc_q, $svc_p);
                if (!$ins_srv_stmt->execute()) {
                    throw new Exception("Insert booking service failed: " . $ins_srv_stmt->error);
                }
            }
            $ins_srv_stmt->close();
        }

        // commit transaction
        $conn->commit();

        // Return success with useful info
        echo json_encode([
            'success' => true,
            'booking_id' => $booking_id,
            'receipt' => [
                'id' => $receipt_id,
                'receipt_number' => $receipt_number,
                'total_amount' => $total_amount,
                'receipt_pdf_path' => $receipt_pdf_path,
                'created_at' => date('Y-m-d H:i:s')
            ],
            'services_saved' => $service_items
        ]);
        exit;

    } catch (Exception $ex) {
        $conn->rollback();
        echo json_encode(['error' => $ex->getMessage()]);
        exit;
    }
}


// --- Recent Bookings (for dashboard) ---
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'recent') {

    $sql = "SELECT 
                b.id,
                b.purpose,
                b.start_time,
                b.end_time,
                b.type,              -- ADD TYPE HERE
                m.name AS member_name
            FROM bookings b
            LEFT JOIN members m ON b.booked_by = m.id
            WHERE b.status = 'booked'
            ORDER BY b.start_time DESC
            LIMIT 5";

    $result = $conn->query($sql);

    $recent = [];
    while ($row = $result->fetch_assoc()) {

        $date = date("Y-m-d", strtotime($row['start_time']));
        $startFmt = date("g:i A", strtotime($row['start_time']));
        $endFmt   = date("g:i A", strtotime($row['end_time']));

        $recent[] = [
            "id" => $row["id"],
            "title" => $row["purpose"] . " - " . $row["member_name"],
            "date" => $date,
            "start_time" => $startFmt,
            "end_time" => $endFmt,
            "type" => $row["type"]   // RETURN TYPE TO FRONTEND
        ];
    }

    echo json_encode($recent);
    exit;
}

// --- List All Bookings (Optionally filter by user/type) ---
if ($method === 'GET' && !isset($_GET['booking_id'])) {
    $type_filter = isset($_GET['type']) ? " AND type = '".$conn->real_escape_string($_GET['type'])."'" : '';
    $sql = "
SELECT 
    b.*,
    m.name AS member_name,
    IFNULL(SUM(p.amount),0) AS paid_amount,
    CASE 
        WHEN IFNULL(SUM(p.amount),0) >= (
            SELECT total_amount 
            FROM booking_receipts 
            WHERE booking_id=b.id 
            ORDER BY id ASC LIMIT 1
        )
        THEN 'paid'
        ELSE 'unpaid'
    END AS payment_status
FROM bookings b
LEFT JOIN members m ON b.booked_by = m.id
LEFT JOIN booking_payments p ON p.booking_id = b.id
WHERE 1 $type_filter
GROUP BY b.id
ORDER BY b.start_time DESC
";

    $result = $conn->query($sql);
    $bookings = [];
    while ($row = $result->fetch_assoc()) {
        // Attach services summary
        $svc_sql = "SELECT service_name, quantity, price FROM booking_services WHERE booking_id=" . $row['id'];
        $svc_result = $conn->query($svc_sql);
        $services = [];
        while ($s = $svc_result->fetch_assoc()) {
            $services[] = $s;
        }
        $row['services'] = $services;
        // Attach total amount from receipt
$amt_sql = "SELECT total_amount FROM booking_receipts 
            WHERE booking_id=" . $row['id'] . " 
            ORDER BY id ASC LIMIT 1";
$amt_res = $conn->query($amt_sql);
$row['total_amount'] = 0;
if ($amt = $amt_res->fetch_assoc()) {
    $row['total_amount'] = floatval($amt['total_amount']);
}

        $bookings[] = $row;
    }
    echo json_encode($bookings);
    exit;
}


// --- Get Booking by ID ---
if ($method === 'GET' && isset($_GET['booking_id'])) {
    $booking_id = intval($_GET['booking_id']);
    $sql = "SELECT b.*, m.name AS member_name FROM bookings b
            LEFT JOIN members m ON b.booked_by = m.id
            WHERE b.id = $booking_id";
    $result = $conn->query($sql);
    if ($row = $result->fetch_assoc()) {
        $svc_sql = "SELECT * FROM booking_services WHERE booking_id=" . $booking_id;
        $svc_result = $conn->query($svc_sql);
        $services = [];
        while ($s = $svc_result->fetch_assoc()) {
            $services[] = $s;
        }
        $row['services'] = $services;
        // Attach receipt(s)
        $r_sql = "SELECT id, booking_id, receipt_number, total_amount, receipt_pdf_path, created_at FROM booking_receipts WHERE booking_id = $booking_id ORDER BY id ASC";
        $r_res = $conn->query($r_sql);
        $receipts = [];
        while ($rr = $r_res->fetch_assoc()) {
            $receipts[] = $rr;
        }
        $row['receipts'] = $receipts;

        // Attach payments
        $pay_sql = "SELECT amount, payment_mode, remarks, paid_at 
                    FROM booking_payments 
                    WHERE booking_id=$booking_id
                    ORDER BY paid_at ASC";
        $pay_res = $conn->query($pay_sql);

        $payments = [];
        $paid_total = 0;
        while ($p = $pay_res->fetch_assoc()) {
            $payments[] = $p;
            $paid_total += floatval($p['amount']);
        }

        $row['payments'] = $payments;
        $row['paid_amount'] = $paid_total;

        // Determine payment status
        $total_amt = 0;
        if (!empty($row['receipts'][0]['total_amount'])) {
            $total_amt = floatval($row['receipts'][0]['total_amount']);
        }
        $row['payment_status'] = ($paid_total >= $total_amt && $total_amt > 0)
            ? 'paid'
            : 'unpaid';


                echo json_encode($row);
            } else {
                echo json_encode(['error' => 'Booking not found']);
            }
            exit;
    }

// --- Update Booking (Only by creator or by role) ---
if ($method === 'PUT' && isset($_GET['booking_id'])) {
    $booking_id = intval($_GET['booking_id']);

    // Only booking creator or admin can edit (extend this with role checks)
    $check_sql = "SELECT booked_by FROM bookings WHERE id = $booking_id";
    $check_result = $conn->query($check_sql);
    if ($check_row = $check_result->fetch_assoc()) {
        if ($check_row['booked_by'] != $user_id) {
            echo json_encode(['error' => 'Not authorized']); exit;
        }
    } else {
        echo json_encode(['error' => 'Booking not found']); exit;
    }

    $input = json_decode(file_get_contents("php://input"), true);
    $purpose = $conn->real_escape_string($input['purpose'] ?? '');
    $start_time = $conn->real_escape_string($input['start_time'] ?? '');
    $end_time = $conn->real_escape_string($input['end_time'] ?? '');
    $status = $conn->real_escape_string($input['status'] ?? 'booked');
    $type = $conn->real_escape_string($input['type'] ?? 'external');

    $sql = "UPDATE bookings SET 
                purpose='$purpose',
                start_time='$start_time',
                end_time='$end_time',
                status='$status',
                type='$type',
                updated_at=NOW()
            WHERE id=$booking_id";
    if ($conn->query($sql)) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['error' => $conn->error]);
    }
    exit;
}

// --- Cancel/Delete Booking (Only by creator or by role) ---
if ($method === 'DELETE' && isset($_GET['booking_id'])) {
    $booking_id = intval($_GET['booking_id']);
    // Only booking creator or admin can cancel
    $check_sql = "SELECT booked_by FROM bookings WHERE id = $booking_id";
    $check_result = $conn->query($check_sql);
    if ($check_row = $check_result->fetch_assoc()) {
        if ($check_row['booked_by'] != $user_id) {
            echo json_encode(['error' => 'Not authorized']); exit;
        }
    } else {
        echo json_encode(['error' => 'Booking not found']); exit;
    }

    // Cancel booking (soft delete)
    $sql = "UPDATE bookings SET status='cancelled', updated_at=NOW() WHERE id=$booking_id";
    if ($conn->query($sql)) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['error' => $conn->error]);
    }
    exit;
}

// --- Add Service to Booking ---
if ($method === 'POST' && isset($_GET['booking_id']) && isset($_GET['service'])) {
    $booking_id = intval($_GET['booking_id']);
    $input = json_decode(file_get_contents("php://input"), true);
    $service_name = $conn->real_escape_string($input['service_name'] ?? '');
    $quantity = intval($input['quantity'] ?? 1);
    $price = floatval($input['price'] ?? 0);

    // Only creator can add services
    $check_sql = "SELECT booked_by FROM bookings WHERE id = $booking_id";
    $check_result = $conn->query($check_sql);
    if ($check_row = $check_result->fetch_assoc()) {
        if ($check_row['booked_by'] != $user_id) {
            echo json_encode(['error' => 'Not authorized']); exit;
        }
    } else {
        echo json_encode(['error' => 'Booking not found']); exit;
    }

    $sql = "INSERT INTO booking_services (booking_id, service_name, quantity, price)
            VALUES ($booking_id, '$service_name', $quantity, $price)";
    if ($conn->query($sql)) {
        echo json_encode(['success' => true, 'service_id' => $conn->insert_id]);
    } else {
        echo json_encode(['error' => $conn->error]);
    }
    exit;
}

// --- Delete Service from Booking ---
if ($method === 'DELETE' && isset($_GET['service_id'])) {
    $service_id = intval($_GET['service_id']);
    // To ensure only creator can delete, join with bookings
    $sql = "DELETE bs FROM booking_services bs
            INNER JOIN bookings b ON bs.booking_id = b.id
            WHERE bs.id=$service_id AND b.booked_by=$user_id";
    if ($conn->query($sql)) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['error' => $conn->error]);
    }
    exit;
}




// --- If No Matching Route ---
http_response_code(400);
echo json_encode(['error' => 'Invalid request']);
?>
