<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../db/db.php';
require_once __DIR__ . '/../jwt_utils/jwt_utils.php';

header("Content-Type: application/json");

// -----------------------------------------
// JWT AUTH (EXACT SAME AS YOUR SAMPLE)
// -----------------------------------------
$headers = getallheaders();

if (empty($headers["Authorization"])) {
    http_response_code(401);
    echo json_encode(["error" => "No token provided"]);
    exit;
}

$jwt = str_replace("Bearer ", "", $headers["Authorization"]);

// OLD METHOD: validate_jwt($jwt, $jwt_secret)
$payload = validate_jwt($jwt, $jwt_secret);

if (!$payload || !isset($payload["id"])) {
    http_response_code(401);
    echo json_encode(["error" => "Invalid token"]);
    exit;
}

$current_user_id = intval($payload["id"]);


require_once __DIR__ . '/access.php';

// IMPORTANT: Fetch user's role_id from JWT
$current_role_id = intval($payload["role_id"]);

// -----------------------------------------
// HELPERS
// -----------------------------------------
function runQuery($conn, $query, $params = [], $types = "")
{
    $stmt = $conn->prepare($query);

    if ($params) {
        if ($types == "") {
            // default all strings
            $types = str_repeat("s", count($params));
        }
        $stmt->bind_param($types, ...$params);
    }

    $stmt->execute();
    return $stmt;
}

// -----------------------------------------
// FIND ACTION
// -----------------------------------------
$action = $_GET["action"] ?? $_POST["action"] ?? "";

// -----------------------------------------
// GET ALL INCOMES
// -----------------------------------------
if ($action === "get_incomes") {

    check_access($conn, $current_role_id, "roles", "view");

    $sql = "
        SELECT 
            id,
            category_id,
            amount,
            description,
            income_date,
            recorded_by,
            created_at,
            'manual' AS source
        FROM incomes

        UNION ALL

        SELECT
            bp.id,
            NULL AS category_id,
            bp.amount,
            CONCAT('Booking Payment #', bp.booking_id) AS description,
            DATE(bp.paid_at) AS income_date,
            bp.paid_by AS recorded_by,
            bp.created_at,
            'booking' AS source
        FROM booking_payments bp

        ORDER BY income_date DESC, id DESC
    ";

    $res = runQuery($conn, $sql)->get_result()->fetch_all(MYSQLI_ASSOC);

    echo json_encode($res);
    exit;
}


// -----------------------------------------
// GET ALL EXPENSES
// -----------------------------------------
if ($action === "get_expenses") {

    check_access($conn, $current_role_id, "roles", "view");

    $sql = "SELECT id, category_id, amount, description, expense_date, recorded_by, created_at 
            FROM expenses ORDER BY expense_date DESC, id DESC";

    $res = runQuery($conn, $sql)->get_result()->fetch_all(MYSQLI_ASSOC);

    echo json_encode($res);
    exit;
}

// -----------------------------------------
// ADD INCOME
// -----------------------------------------
if ($action === "add_income") {

    // Require add permission
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "add");

    $category_id = $_POST["category_id"] ?? null;
    if ($category_id === "" || $category_id === "0") { 
        $category_id = null; 
    }
    $amount      = $_POST["amount"] ?? 0;
    $description = $_POST["description"] ?? "";
    $income_date = $_POST["income_date"] ?? date("Y-m-d");

    $sql = "INSERT INTO incomes (category_id, amount, description, income_date, recorded_by, created_at)
            VALUES (?, ?, ?, ?, ?, NOW())";

    runQuery($conn, $sql, [$category_id, $amount, $description, $income_date, $current_user_id], "idssi");

    echo json_encode(["status" => true, "message" => "Income added"]);
    exit;
}

// -----------------------------------------
// ADD EXPENSE
// -----------------------------------------
if ($action === "add_expense") {

    // Require add permission
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "add");

    $category_id = $_POST["category_id"] ?? null;
    if ($category_id === "" || $category_id === "0") { 
        $category_id = null; 
    }

    $amount      = $_POST["amount"] ?? 0;
    $description = $_POST["description"] ?? "";
    $expense_date = $_POST["expense_date"] ?? date("Y-m-d");

    $sql = "INSERT INTO expenses (category_id, amount, description, expense_date, recorded_by, created_at)
            VALUES (?, ?, ?, ?, ?, NOW())";

    runQuery($conn, $sql, [$category_id, $amount, $description, $expense_date, $current_user_id], "idssi");

    echo json_encode(["status" => true, "message" => "Expense added"]);
    exit;
}

// -----------------------------------------
// DELETE INCOME
// -----------------------------------------
if ($action === "delete_income") {

    // Require add + edit permission
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "edit");
    check_access($conn, $current_role_id, "roles", "add");

    $id = $_GET["id"] ?? "";

    if (!$id) {
        echo json_encode(["status" => false, "message" => "ID required"]);
        exit;
    }

    runQuery($conn, "DELETE FROM incomes WHERE id=?", [$id], "i");

    echo json_encode(["status" => true, "message" => "Income deleted"]);
    exit;
}

// -----------------------------------------
// DELETE EXPENSE
// -----------------------------------------
if ($action === "delete_expense") {

    // Require add + edit permission
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "edit");
    check_access($conn, $current_role_id, "roles", "add");

    $id = $_GET["id"] ?? "";

    if (!$id) {
        echo json_encode(["status" => false, "message" => "ID required"]);
        exit;
    }

    runQuery($conn, "DELETE FROM expenses WHERE id=?", [$id], "i");

    echo json_encode(["status" => true, "message" => "Expense deleted"]);
    exit;
}

// -----------------------------------------
// FILTER BY DATE RANGE
// -----------------------------------------
if ($action === "filter") {

    check_access($conn, $current_role_id, "roles", "view");

    $from = $_GET["from"] ?? "";
    $to   = $_GET["to"] ?? "";

    if (!$from || !$to) {
        echo json_encode(["status" => false, "message" => "Date range required"]);
        exit;
    }

    // Income filter
$sql_inc = "
    SELECT 
        id, category_id, amount, description, income_date, recorded_by, created_at, 'manual' AS source
    FROM incomes
    WHERE income_date BETWEEN ? AND ?

    UNION ALL

    SELECT
        bp.id,
        NULL,
        bp.amount,
        CONCAT('Booking Payment #', bp.booking_id),
        DATE(bp.paid_at),
        bp.paid_by,
        bp.created_at,
        'booking'
    FROM booking_payments bp
    WHERE DATE(bp.paid_at) BETWEEN ? AND ?

    ORDER BY income_date DESC
";

$income = runQuery($conn, $sql_inc, [$from, $to, $from, $to], "ssss")
            ->get_result()
            ->fetch_all(MYSQLI_ASSOC);

    // Expense filter
    $sql_exp = "SELECT * FROM expenses WHERE expense_date BETWEEN ? AND ? ORDER BY expense_date DESC";
    $expense = runQuery($conn, $sql_exp, [$from, $to])->get_result()->fetch_all(MYSQLI_ASSOC);

    echo json_encode([
        "income" => $income,
        "expenses" => $expense
    ]);
    exit;
}

// ------------------------------------------------------
// EXPORT CSV
// ------------------------------------------------------
if ($action === "export_csv") {

    check_access($conn, $current_role_id, "roles", "view");

    $from = $_GET['from'] ?? "";
    $to   = $_GET['to'] ?? "";

    header("Content-Type: text/csv");
    header("Content-Disposition: attachment; filename=finance_export.csv");

    $output = fopen("php://output", "w");

    // CSV header
    fputcsv($output, ["Type", "Category", "Amount", "Description", "Date"]);

    // --- BOOKING PAYMENTS AS INCOME ---
$sql = "
    SELECT bp.amount, bp.paid_at
    FROM booking_payments bp
";

if ($from && $to) {
    $sql .= " WHERE DATE(bp.paid_at) BETWEEN ? AND ?";
    $stmt = runQuery($conn, $sql, [$from, $to], "ss");
} else {
    $stmt = runQuery($conn, $sql);
}

$bookingIncome = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

foreach ($bookingIncome as $row) {
    fputcsv($output, [
        "Income",
        "Booking Payment",
        $row["amount"],
        "Booking payment received",
        $row["paid_at"]
    ]);
}
    // --- EXPENSES ---
    $sql = "SELECT expenses.amount, expenses.description, expenses.expense_date, 
                   expense_categories.category_name 
            FROM expenses
            LEFT JOIN expense_categories ON expenses.category_id = expense_categories.id";

    if ($from && $to) {
        $sql .= " WHERE expense_date BETWEEN ? AND ?";
        $stmt = runQuery($conn, $sql, [$from, $to], "ss");
    } else {
        $stmt = runQuery($conn, $sql);
    }

    $expenseRows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

    foreach ($expenseRows as $row) {
        fputcsv($output, [
            "Expense",
            $row["category_name"],
            $row["amount"],
            $row["description"],
            $row["expense_date"]
        ]);
    }

    fclose($output);
    exit;
}


// -----------------------------------------
// INVALID ACTION
// -----------------------------------------
echo json_encode(["status" => false, "message" => "Invalid action"]);
exit;

?>
