<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../db/db.php';
require_once __DIR__ . '/../jwt_utils/jwt_utils.php';

header("Content-Type: application/json");

// -----------------------------
// AUTH CHECK
// -----------------------------
$headers = getallheaders();
if (empty($headers['Authorization'])) {
    http_response_code(401);
    echo json_encode(["error" => "No token"]);
    exit;
}
$jwt = str_replace("Bearer ", "", $headers["Authorization"]);
$payload = validate_jwt($jwt, $jwt_secret);
if (!$payload || !isset($payload["id"])) {
    http_response_code(401);
    echo json_encode(["error" => "Invalid token"]);
    exit;
}
$current_user_id = intval($payload["id"]);
$method = $_SERVER["REQUEST_METHOD"];

// -----------------------------
// Helpers
// -----------------------------
function fetch_row($conn, $sql, $types = "", $params = []) {
    $stmt = $conn->prepare($sql);
    if ($stmt === false) return null;
    if ($types !== "") $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;
    $stmt->close();
    return $row;
}

function get_json_body() {
    $data = json_decode(file_get_contents("php://input"), true);
    return is_array($data) ? $data : [];
}

// -----------------------------
// LIST MEETINGS
// -----------------------------
if ($method === "GET" && ($_GET["action"] ?? "") === "list") {

    // Get total members once (same for all meetings)
    $totalMembersQuery = $conn->query("SELECT COUNT(id) AS total_members FROM members");
    $totalMembers = $totalMembersQuery->fetch_assoc()["total_members"] ?? 0;

    $sql = "
        SELECT 
            b.id AS booking_id, b.booked_by, b.purpose, b.start_time, b.end_time,
            b.status, b.type,
            b.created_at AS booking_created_at, b.updated_at AS booking_updated_at,
            m.id AS meeting_id, m.meeting_date, 
            m.created_by AS meeting_created_by, 
            m.created_at AS meeting_created_at
        FROM bookings b
        LEFT JOIN meetings m ON m.booking_id = b.id
        ORDER BY b.start_time DESC
    ";

    $res = $conn->query($sql);
    $out = [];

    while ($row = $res->fetch_assoc()) {

        $meeting_id = intval($row["meeting_id"]);

        // Count total attendance required (meeting_attendance table)
        $stmt1 = $conn->prepare("SELECT COUNT(*) AS required_attendance FROM meeting_attendance WHERE meeting_id = ?");
        $stmt1->bind_param("i", $meeting_id);
        $stmt1->execute();
        $req = $stmt1->get_result()->fetch_assoc();
        $stmt1->close();

        // Count meeting minutes
        $stmt2 = $conn->prepare("SELECT COUNT(*) AS cnt FROM meeting_minutes WHERE meeting_id = ?");
        $stmt2->bind_param("i", $meeting_id);
        $stmt2->execute();
        $min = $stmt2->get_result()->fetch_assoc();
        $stmt2->close();

        // Add new fields
        $row["required_attendance"] = intval($req["required_attendance"]);
        $row["total_members"] = intval($totalMembers);
        $row["minutes_count"] = intval($min["cnt"]);

        $out[] = $row;
    }

    echo json_encode($out);
    exit;
}

// -----------------------------
// GET SINGLE MEETING
// -----------------------------
if ($method === "GET" && isset($_GET["id"]) && !isset($_GET["action"])) {

    $meeting_id = intval($_GET["id"]);

    $sql = "
        SELECT 
            b.id AS booking_id, b.booked_by, b.purpose, b.start_time, b.end_time,
            b.status, b.type,
            m.id AS meeting_id, m.meeting_date
        FROM meetings m
        JOIN bookings b ON b.id = m.booking_id
        WHERE m.id = ? LIMIT 1
    ";
    $row = fetch_row($conn, $sql, "i", [$meeting_id]);
    if (!$row) {
        echo json_encode(["error" => "Meeting not found"]);
        exit;
    }

    $stmt = $conn->prepare("SELECT id, minutes_text, created_by, created_at FROM meeting_minutes WHERE meeting_id = ? ORDER BY created_at ASC");
    $stmt->bind_param("i", $meeting_id);
    $stmt->execute();
    $res = $stmt->get_result();
    $minutes = [];
    while ($m = $res->fetch_assoc()) $minutes[] = $m;
    $stmt->close();

    $row["minutes"] = $minutes;

    echo json_encode($row);
    exit;
}

// -----------------------------
// GET MINUTES
// -----------------------------
if ($method === "GET" && ($_GET["action"] ?? "") === "minutes" && isset($_GET["id"])) {

    $meeting_id = intval($_GET["id"]);

    $stmt = $conn->prepare("SELECT id, minutes_text, created_by, created_at FROM meeting_minutes WHERE meeting_id = ? ORDER BY created_at ASC");
    $stmt->bind_param("i", $meeting_id);
    $stmt->execute();
    $res = $stmt->get_result();
    $out = [];
    while ($r = $res->fetch_assoc()) $out[] = $r;

    $stmt->close();
    echo json_encode($out);
    exit;
}

// -----------------------------
// QR GENERATION
// -----------------------------
if ($method === "GET" && ($_GET["action"] ?? "") === "qr" && isset($_GET["id"])) {

    require_once __DIR__ . '/../libs/phpqrcode/qrlib.php';

    $meeting_id = intval($_GET["id"]);

    $row = fetch_row($conn, "
        SELECT m.id FROM meetings m WHERE m.id = ? LIMIT 1
    ", "i", [$meeting_id]);

    if (!$row) {
        echo json_encode(["error" => "Meeting not found"]);
        exit;
    }

    $qr_content = "meeting_id:" . $meeting_id;

    $qrDir = __DIR__ . '/../qr/';
    if (!is_dir($qrDir)) mkdir($qrDir, 0777, true);

    $file = $qrDir . "meeting_" . $meeting_id . ".png";
    QRcode::png($qr_content, $file, QR_ECLEVEL_H, 8, 2);

    $baseUrl =
        (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https://" : "http://")
        . $_SERVER['HTTP_HOST']
        . dirname($_SERVER['REQUEST_URI'], 2);

    echo json_encode([
        "meeting_id" => $meeting_id,
        "qr_url" => $baseUrl . "/qr/meeting_" . $meeting_id . ".png"
    ]);
    exit;
}

// -----------------------------
// CREATE MEETING
// -----------------------------
if ($method === "POST" && !isset($_GET["id"]) && !isset($_GET["action"])) {

    $input = get_json_body();

    $title = trim($input["title"] ?? "");
    $date  = trim($input["date"] ?? "");
    $start = trim($input["start"] ?? "");
    $end   = trim($input["end"] ?? "");

    if ($title === "" || $date === "" || $start === "" || $end === "") {
        echo json_encode(["error" => "Missing fields"]);
        exit;
    }

    $start_dt = "$date $start:00";
    $end_dt   = "$date $end:00";

    // 1-HOUR BUFFER CHECK
    $sql = "
        SELECT id, purpose, start_time, end_time
        FROM bookings
        WHERE start_time < DATE_ADD(?, INTERVAL 1 HOUR)
          AND end_time   > DATE_SUB(?, INTERVAL 1 HOUR)
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $end_dt, $start_dt);
    $stmt->execute();
    $res = $stmt->get_result();

    if ($conf = $res->fetch_assoc()) {
        echo json_encode([
            "error" => "Time slot not available",
            "conflict" => $conf
        ]);
        exit;
    }
    $stmt->close();

    // INSERT NOW
    $conn->begin_transaction();
    try {
        $stmt = $conn->prepare("
            INSERT INTO bookings (booked_by, purpose, start_time, end_time, status, type, created_at, updated_at)
            VALUES (?, ?, ?, ?, 'booked', 'internal', NOW(), NOW())
        ");
        $stmt->bind_param("isss", $current_user_id, $title, $start_dt, $end_dt);
        $stmt->execute();
        $booking_id = $conn->insert_id;

        $stmt->close();

        $stmt = $conn->prepare("
            INSERT INTO meetings (booking_id, meeting_date, created_by, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        $stmt->bind_param("isi", $booking_id, $date, $current_user_id);
        $stmt->execute();
        $meeting_id = $conn->insert_id;

        $stmt->close();

        $conn->commit();

        echo json_encode([
            "success" => true,
            "booking_id" => $booking_id,
            "meeting_id" => $meeting_id
        ]);

    } catch (Exception $e) {
        $conn->rollback();
        echo json_encode(["error" => $e->getMessage()]);
    }

    exit;
}

// -----------------------------
// CHECK AVAILABILITY (WITH 1 HR BUFFER)
// -----------------------------
if ($method === "POST" && ($_GET["action"] ?? "") === "check") {

    $input = get_json_body();

    $date  = trim($input["date"] ?? "");
    $start = trim($input["start"] ?? "");
    $end   = trim($input["end"] ?? "");

    $start_dt = "$date $start:00";
    $end_dt   = "$date $end:00";

    $sql = "
        SELECT id, purpose, start_time, end_time
        FROM bookings
        WHERE start_time < DATE_ADD(?, INTERVAL 1 HOUR)
          AND end_time   > DATE_SUB(?, INTERVAL 1 HOUR)
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ss", $end_dt, $start_dt);
    $stmt->execute();
    $res = $stmt->get_result();

    $conflicts = [];
    while ($r = $res->fetch_assoc()) $conflicts[] = $r;

    echo json_encode([
        "available" => count($conflicts) === 0,
        "conflicts" => $conflicts
    ]);
    exit;
}

// -----------------------------
// ADD MINUTES
// -----------------------------
if ($method === "POST" && ($_GET["action"] ?? "") === "minutes" && isset($_GET["id"])) {

    $meeting_id = intval($_GET["id"]);
    $input = get_json_body();
    $text = trim($input["minutes_text"] ?? "");

    if ($text === "") {
        echo json_encode(["error" => "minutes_text required"]);
        exit;
    }

    $stmt = $conn->prepare("
        INSERT INTO meeting_minutes (meeting_id, minutes_text, created_by, created_at)
        VALUES (?, ?, ?, NOW())
    ");
    $stmt->bind_param("isi", $meeting_id, $text, $current_user_id);
    $stmt->execute();
    $id = $stmt->insert_id;
    $stmt->close();

    echo json_encode(["success" => true, "minutes_id" => $id]);
    exit;
}

// -----------------------------
// GET MINUTES (Single latest or empty)
// -----------------------------
if ($method === "GET" && ($_GET["action"] ?? "") === "get_minutes" && isset($_GET["id"])) {

    $meeting_id = intval($_GET["id"]);

    $stmt = $conn->prepare("SELECT minutes_text FROM meeting_minutes WHERE meeting_id = ? ORDER BY created_at DESC LIMIT 1");
    $stmt->bind_param("i", $meeting_id);
    $stmt->execute();

    $res = $stmt->get_result();
    $row = $res->fetch_assoc();

    echo json_encode([
        "success" => true,
        "minutes_text" => $row["minutes_text"] ?? ""
    ]);
    exit;
}

// -----------------------------
// UPDATE MEETING (PUT)
// -----------------------------
if ($method === "PUT" && isset($_GET["id"])) {

    $meeting_id = intval($_GET["id"]);
    $input = get_json_body();

    $title = trim($input["title"] ?? "");
    $minutes = trim($input["minutes_text"] ?? "");

    if ($title === "") {
        echo json_encode(["error" => "Title required"]);
        exit;
    }

    // ---------------------------------------------------------
    // Fetch booking_id + original start_time (to check if editable)
    // ---------------------------------------------------------
    $stmt = $conn->prepare("
        SELECT b.id AS booking_id, b.start_time 
        FROM meetings m 
        JOIN bookings b ON b.id = m.booking_id
        WHERE m.id = ? LIMIT 1
    ");
    $stmt->bind_param("i", $meeting_id);
    $stmt->execute();
    $info = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    if (!$info) {
        echo json_encode(["error" => "Meeting not found"]);
        exit;
    }

    $booking_id = intval($info["booking_id"]);
    $original_start = new DateTime($info["start_time"]);
    $now = new DateTime();

    $can_edit_time = ($original_start > $now);   // time editable only BEFORE meeting start

    // ---------------------------------------------------------
    // If date/time are included and editable → validate and update them
    // ---------------------------------------------------------
    if ($can_edit_time && isset($input["date"]) && isset($input["start"]) && isset($input["end"])) {

        $date  = trim($input["date"]);
        $start = trim($input["start"]);
        $end   = trim($input["end"]);

        if ($date === "" || $start === "" || $end === "") {
            echo json_encode(["error" => "Missing date/time fields"]);
            exit;
        }

        $new_start_dt = "$date $start:00";
        $new_end_dt   = "$date $end:00";

        // Availability buffer check
        $sql = "
            SELECT id, purpose, start_time, end_time
            FROM bookings
            WHERE id != ?
              AND start_time < DATE_ADD(?, INTERVAL 1 HOUR)
              AND end_time   > DATE_SUB(?, INTERVAL 1 HOUR)
        ";

        $stmt = $conn->prepare($sql);
        $stmt->bind_param("iss", $booking_id, $new_end_dt, $new_start_dt);
        $stmt->execute();
        $conf = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if ($conf) {
            echo json_encode([
                "error" => "Time slot not available",
                "conflict" => [
                    "purpose" => $conf["purpose"],
                    "start_time" => $conf["start_time"],
                    "end_time" => $conf["end_time"]
                ]
            ]);
            exit;
        }

        // Update booking
        $stmt = $conn->prepare("
            UPDATE bookings 
            SET purpose = ?, start_time = ?, end_time = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->bind_param("sssi", $title, $new_start_dt, $new_end_dt, $booking_id);
        $stmt->execute();
        $stmt->close();

        // Update meeting date
        $stmt = $conn->prepare("UPDATE meetings SET meeting_date = ? WHERE id = ?");
        $stmt->bind_param("si", $date, $meeting_id);
        $stmt->execute();
        $stmt->close();

    } else {
        // Only update title (after meeting already started)
        $stmt = $conn->prepare("UPDATE bookings SET purpose = ?, updated_at = NOW() WHERE id = ?");
        $stmt->bind_param("si", $title, $booking_id);
        $stmt->execute();
        $stmt->close();
    }

    // ---------------------------------------------------------
    // UPDATE / INSERT MINUTES (Always allowed after meeting start)
    // ---------------------------------------------------------
    if ($minutes !== "") {

        $stmt = $conn->prepare("SELECT id FROM meeting_minutes WHERE meeting_id = ? LIMIT 1");
        $stmt->bind_param("i", $meeting_id);
        $stmt->execute();
        $exists = $stmt->get_result()->fetch_assoc();
        $stmt->close();

        if ($exists) {
            $stmt = $conn->prepare("
                UPDATE meeting_minutes 
                SET minutes_text = ?, created_at = NOW() 
                WHERE meeting_id = ?
            ");
            $stmt->bind_param("si", $minutes, $meeting_id);
        } else {
            $stmt = $conn->prepare("
                INSERT INTO meeting_minutes (meeting_id, minutes_text, created_by, created_at)
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->bind_param("isi", $meeting_id, $minutes, $current_user_id);
        }
        $stmt->execute();
        $stmt->close();
    }

    echo json_encode(["success" => true]);
    exit;
}







echo json_encode(["error" => "Invalid request"]);
exit;
?>
