<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../db/db.php';
require_once __DIR__ . '/../jwt_utils/jwt_utils.php';

header("Content-Type: application/json");

// -----------------------------------------
// JWT AUTH CHECK
// -----------------------------------------
$headers = getallheaders();

if (empty($headers['Authorization'])) {
    http_response_code(401);
    echo json_encode(["error" => "No token"]);
    exit;
}

$jwt = str_replace("Bearer ", "", $headers['Authorization']);
$payload = validate_jwt($jwt, $jwt_secret);

if (!$payload) {
    http_response_code(401);
    echo json_encode(["error" => "Invalid token"]);
    exit;
}

require_once __DIR__ . '/access.php';

// IMPORTANT: Fetch user's role_id from JWT
$current_role_id = intval($payload["role_id"]);

// ACTION ROUTER
$action = $_GET['action'] ?? '';

switch ($action) {
    case 'get':
        getServices($conn);
        break;

    case 'add':
        addService($conn);
        break;

    case 'edit':
        editService($conn);
        break;

    case 'delete':
        deleteService($conn);
        break;

    default:
        echo json_encode(["error" => "Invalid action"]);
        break;
}

/* ==========================================================
   FUNCTIONS
   ========================================================== */

// ------------------------------------------------------------
// GET ALL SERVICES
// ------------------------------------------------------------
function getServices($conn)
{
    global $current_role_id;
    check_access($conn, $current_role_id, "roles", "view");

    $sql = "SELECT * FROM services ORDER BY id ASC";
    $result = $conn->query($sql);

    $services = [];
    while ($row = $result->fetch_assoc()) {
        $services[] = $row;
    }

    echo json_encode([
        "status" => true,
        "data" => $services
    ]);
    exit;
}

// ------------------------------------------------------------
// ADD NEW SERVICE
// ------------------------------------------------------------
function addService($conn)
{
    // Require add permission
    global $current_role_id;
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "add");

    $input = json_decode(file_get_contents("php://input"), true);

    if (empty($input['service_name']) || empty($input['price'])) {
        http_response_code(400);
        echo json_encode(["error" => "service_name and price are required"]);
        exit;
    }

    $sql = "INSERT INTO services (service_name, price) VALUES (?, ?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("si", $input['service_name'], $input['price']);

    if ($stmt->execute()) {
        echo json_encode([
            "status" => true,
            "message" => "Service added successfully",
            "service_id" => $stmt->insert_id
        ]);
    } else {
        echo json_encode(["error" => "Failed to add service"]);
    }
    exit;
}

// ------------------------------------------------------------
// EDIT SERVICE
// ------------------------------------------------------------
function editService($conn)
{
    // Require add + edit permission
    global $current_role_id;
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "edit");
    check_access($conn, $current_role_id, "roles", "add");

    $input = json_decode(file_get_contents("php://input"), true);

    if (empty($input['id']) || empty($input['service_name']) || empty($input['price'])) {
        http_response_code(400);
        echo json_encode(["error" => "id, service_name, price required"]);
        exit;
    }

    $sql = "UPDATE services SET service_name=?, price=? WHERE id=?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("sii", $input['service_name'], $input['price'], $input['id']);

    if ($stmt->execute()) {
        echo json_encode(["status" => true, "message" => "Service updated successfully"]);
    } else {
        echo json_encode(["error" => "Failed to update service"]);
    }
    exit;
}

// ------------------------------------------------------------
// DELETE SERVICE
// ------------------------------------------------------------
function deleteService($conn)
{
    // Require add + edit permission
    global $current_role_id;
    check_access($conn, $current_role_id, "roles", "view");
    check_access($conn, $current_role_id, "roles", "edit");
    check_access($conn, $current_role_id, "roles", "add");

    $input = json_decode(file_get_contents("php://input"), true);

    if (empty($input['id'])) {
        http_response_code(400);
        echo json_encode(["error" => "id required"]);
        exit;
    }

    $sql = "DELETE FROM services WHERE id=?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $input['id']);

    if ($stmt->execute()) {
        echo json_encode(["status" => true, "message" => "Service deleted successfully"]);
    } else {
        echo json_encode(["error" => "Failed to delete service"]);
    }
    exit;
}
?>
